<?php
/**
 *
 * This file is part of the phpBB Forum Software package.
 *
 * @copyright (c) phpBB Limited <https://www.phpbb.com>
 * @license GNU General Public License, version 2 (GPL-2.0)
 *
 * For full copyright and license information, please see
 * the docs/CREDITS.txt file.
 *
 */

namespace phpbb\template\twig\extension;

use phpbb\avatar\helper;
use phpbb\template\twig\environment;
use Twig\Error\Error;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

class avatar extends AbstractExtension
{
	/**
	 * @var helper
	 */
	private $avatar_helper;

	/**
	 * Constructor for avatar extension
	 *
	 * @param helper $avatar_helper
	 */
	public function __construct(helper $avatar_helper)
	{
		$this->avatar_helper = $avatar_helper;
	}

	/**
	 * Get the name of this extension
	 *
	 * @return string
	 */
	public function getName(): string
	{
		return 'avatar';
	}

	/**
	 * Returns a list of global functions to add to the existing list.
	 *
	 * @return TwigFunction[] An array of global functions
	 */
	public function getFunctions(): array
	{
		return [
			new TwigFunction('avatar', [$this, 'get_avatar'], ['needs_environment' => true]),
		];
	}

	/**
	 * Get avatar for placing into templates.
	 *
	 * How to use in a template:
	 * - {{ avatar(row, alt, ignore_config, lazy) }}
	 *
	 * The mode and row (group_row or user_row) are required.
	 * The other fields (alt|ignore_config|lazy) are optional.
	 *
	 * @return string	The avatar HTML for the specified mode
	 */
	public function get_avatar(environment $environment, array $row): string
	{
		try
		{
			// Quickly return if required data is missing
			if (!isset($row['src'], $row['width'], $row['height'], $row['title']))
			{
				return '';
			}

			return $environment->render('macros/avatar.twig', [
				'SRC'		=> !empty($row['lazy']) ? $this->avatar_helper->get_no_avatar_source() : $row['src'],
				'DATA_SRC'	=> !empty($row['lazy']) ? $row['src'] : '',
				'WIDTH'		=> $row['width'],
				'HEIGHT'	=> $row['height'],
				'TITLE'		=> $row['title'],
				'LAZY'		=> $row['lazy'] ?? false,
			]);
		}
		catch (Error $e)
		{
			return '';
		}
	}
}
